/**************************************************************************
 * Eliza - The classic artificial intelligence therapist                  *
 * Creative Computing                                                     *
 * Morristown, New Jersey                                                 *
 * Adapted for IBM PC (in BASIC) by Patricia Danielson and Paul Hashfield *
 * Ported to C for maximum compatibility by James Williams                *
 * Copyright (C) 1995  James Williams                                     *
 *                                                                        *
 * This program is free software; you can redistribute it and/or modify   *
 * it under the terms of the GNU General Public License as published by   *
 * the Free Software Foundation; either version 2 of the License, or      *
 * (at your option) any later version.                                    *
 *                                                                        *
 * This program is distributed in the hope that it will be useful,        *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          *
 * GNU General Public License for more details.                           *
 *                                                                        *
 * You should have received a copy of the GNU General Public License      *
 * along with this program; if not, write to the Free Software            *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              *
 **************************************************************************/


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#define isws(x)  ((x) == ' ' || (x) == '\t' || (x) == '\n')

void init(void);
void exchange_words(char *string, int word_start, int word_len,
                    char *new_word);
void ucase(char *string);
void reformat_input(void);
int get_input(void);
char *find_keyword(void);
char *conjugate(char *found);
void response(char *tail);


#define MAX_BUFFER 255
char input_string_buffer[MAX_BUFFER+4] = "  ";
char *input_string = input_string_buffer+2;
char last_input_string[MAX_BUFFER];

char *keyword[] =
{
    "THANK", "CAN YOU ", "CAN I ", "YOU ARE ", "YOU'RE ", "I DON'T ",
    "I FEEL ", "WHY DON'T YOU ", "WHY CAN'T I ", "ARE YOU ",
    "I CAN'T ", "I AM ", "I'M ", "YOUR ", "I WANT ", "WHAT ",
    "HOW ", "WHO ", "WHERE ", "WHEN ", "WHY ", "NAME", "CAUSE ",
    "SORRY ", "DREAM", "HELLO ", "HI ", "MAYBE ", "NO ", "NOT ",
    "NOTHING ", "YOU ", "ALWAYS ", "THINK ", "ALIKE ", "YES ",
    "FRIEND ", "COMPUTER", "NOKEYFOUND"
};

char *wordin[] =
{
    " ARE ", " WERE ", " YOUR ", " I'VE ", " I'M ", " ME ",
    " AM ", " WAS ", " I ", " MY ", " YOU'VE ", " YOU'RE ", " YOU "
};
char *wordout[] =
{
    " AM ", " WAS ", " MY ", " YOU'VE ", " YOU'RE ", " YOU ",
    " ARE ", " WERE ", " YOU ", " YOUR ", " I'VE ", " I'M ", " ME "
};

char *replies[] =
{
    /* THANK */
    "YOU'RE WELCOME",
    "I'M GLAD I COULD HELP",

    /* CAN YOU */
    "DON'T YOU BELIEVE THAT I CAN*",
    "PERHAPS YOU BELIEVE I SHOULD*",
    "YOU WANT ME TO BE ABLE TO*",

    /* CAN I */
    "PERHAPS YOU DON'T WANT TO*",
    "DO YOU WANT TO BE ABLE TO*",

    /* YOU ARE, YOU'RE */
    "WHAT MAKES YOU THINK I AM*",
    "DOES IT PLEASE YOU TO BELIEVE I AM*",
    "PERHAPS YOU WOULD LIKE TO BE*",
    "DO YOU SOMETIMES WISH YOU WERE*",

    /* I DON'T */
    "DON'T YOU REALLY*",
    "WHY DON'T YOU*",
    "DO YOU WISH TO BE ABLE TO*",
    "DOES IT TROUBLE YOU TO NOT*",

    /* I FEEL */
    "DO YOU OFTEN FEEL*",
    "WHY DO YOU THINK YOU FEEL*",
    "DO YOU ENJOY FEELING*",

    /* WHY DON'T YOU */
    "DO YOU REALLY BELIEVE I DON'T*",
    "PERHAPS IN GOOD TIME I WILL*",
    "DO YOU WANT ME TO*",

    /* WHY CAN'T I */
    "DO YOU THINK YOU SHOULD BE ABLE TO*",
    "WHY CAN'T YOU*",

    /* ARE YOU */
    "WHY ARE YOU INTERESTED IN WHETHER OR NOT I AM*",
    "WOULD YOU PREFER IF I WERE NOT*",
    "PERHAPS IN YOUR FANTASIES I AM*",

    /* I CAN'T */
    "HOW DO YOU KNOW YOU CAN'T*",
    "HAVE YOU TRIED?",
    "PERHAPS YOU CAN NOW*",

    /* I AM, I'M */
    "DID YOU COME TO ME BECAUSE YOU ARE*",
    "HOW LONG HAVE YOU BEEN*",
    "DO YOU BELIEVE IT IS NORMAL TO BE*",
    "DO YOU ENJOY BEING*",

    /* YOUR */
    "WHY ARE YOU CONCERNED ABOUT MY*",
    "WHAT ABOUT YOUR OWN*",

    /* I WANT */
    "WHAT WOULD IT MEAN TO YOU IF YOU GOT*",
    "WHY DO YOU WANT*",
    "SUPPOSE YOU SOON GOT*",
    "WHAT IF YOU NEVER GOT*",
    "I SOMETIMES ALSO WANT*",

    /* HOW, WHO, WHERE, WHEN, WHY */
    "WHY DO YOU ASK?",
    "DOES THAT QUESTION INTEREST YOU?",
    "WHAT ANSWER WOULD PLEASE YOU THE MOST?",
    "WHAT DO YOU THINK?",
    "ARE SUCH QUESTIONS ON YOUR MIND OFTEN?",
    "WHAT IS IT THAT YOU REALLY WANT TO KNOW?",
    "HAVE YOU ASKED ANYONE ELSE?",
    "HAVE YOU ASKED SUCH QUESTIONS BEFORE?",
    "WHAT else COMES TO MIND WHEN YOU ASK THAT?",

    /* NAME */
    "NAMES DON'T INTEREST ME.",
    "I DON'T CARE ABOUT NAMES --PLEASE GO ON.",

    /* CAUSE */
    "IS THAT THE REAL REASON?",
    "DON'T ANY OTHER REASONS COME TO MIND?",
    "DOES THAT REASON EXPLAIN ANYTHING ELSE?",
    "WHAT OTHER REASONS MIGHT THERE BE?",

    /* SORRY */
    "PLEASE DON'T APOLOGIZE!",
    "APOLOGIES ARE NOT NECESSARY.",
    "WHAT FEELINGS DO YOU HAVE WHEN YOU APOLOGIZE?",
    "DON'T BE SO DEFENSIVE!",

    /* DREAM */
    "WHAT DOES THAT DREAM SUGGEST TO YOU?",
    "DO YOU DREAM OFTEN?",
    "WHAT PERSONS APPEAR IN YOUR DREAMS?",
    "ARE YOU DISTURBED BY YOUR DREAMS?",

    /* HELLO, HI */
    "HOW DO YOU DO ...PLEASE STATE YOUR PROBLEM.",

    /* MAYBE */
    "YOU DON'T SEEM QUITE CERTAIN.",
    "WHY THE UNCERTAIN TONE?",
    "CAN'T YOU BE MORE POSITIVE?",
    "YOU AREN'T SURE?",
    "DON'T YOU KNOW?",

    /* NO, NOT, NOTHING */
    "ARE YOU SAYING NO JUST TO BE NEGATIVE?",
    "YOU ARE BEING A BIT NEGATIVE.",
    "WHY NOT?",
    "ARE YOU SURE?",
    "WHY NO?",

    /* YOU */
    "WE WERE DISCUSSING YOU--NOT ME.",
    "OH, I*",
    "YOU'RE NOT REALLY TALKING ABOUT ME, ARE YOU?",

    /* ALWAYS */
    "CAN YOU THINK OF A SPECIFIC EXAMPLE?",
    "WHEN?",
    "WHAT ARE YOU THINKING OF?",
    "REALLY, ALWAYS?",

    /* THINK */
    "DO YOU REALLY THINK SO?",
    "BUT YOU ARE NOT SURE*",
    "DO YOU DOUBT*",

    /* ALIKE */
    "IN WHAT WAY?",
    "WHAT RESEMBLANCE DO YOU SEE?",
    "WHAT DOES THE SIMILARITY SUGGEST TO YOU?",
    "WHAT OTHER CONNECTIONS DO YOU SEE?",
    "COULD THERE REALLY BE SOME CONNECTION?",
    "HOW?",

    /* YES */
    "ARE YOU SURE?",
    "I SEE.",
    "I UNDERSTAND.",
    "YOU SEEM QUITE POSITIVE.",

    /* FRIEND */
    "WHY DO YOU BRING UP THE TOPIC OF FRIENDS?",
    "DO YOUR FRIENDS WORRY YOU?",
    "DO YOUR FRIENDS PICK ON YOU?",
    "ARE YOU SURE YOU HAVE ANY FRIENDS?",
    "DO YOU IMPOSE ON YOUR FRIENDS?",
    "PERHAPS YOUR LOVE FOR FRIENDS WORRIES YOU.",

    /* COMPUTER */
    "DO COMPUTERS WORRY YOU?",
    "ARE YOU TALKING ABOUT ME IN PARTICULAR?",
    "ARE YOU FRIGHTENED BY MACHINES?",
    "WHY DO YOU MENTION COMPUTERS?",
    "WHAT DO YOU THINK MACHINES HAVE TO DO WITH YOUR PROBLEM?",
    "DON'T YOU THINK COMPUTERS CAN HELP PEOPLE?",
    "WHAT IS IT ABOUT MACHINES THAT WORRIES YOU?",

    /* NOKEYFOUND */
    "WHAT DOES THAT SUGGEST TO YOU?",
    "I SEE.",
    "I'M NOT SURE I UNDERSTAND YOU FULLY.",
    "COME COME ELUCIDATE YOUR THOUGHTS.",
    "CAN YOU ELABORATE ON THAT?",
    "THAT IS QUITE INTERESTING.",
    "SAY, DO YOU HAVE ANY PSYCHOLOGICAL PROBLEMS?"
};


int num_of_responses[] = {
/* This tells how many responses are available for each keyword.  A -1
 * means the word is synonymous with the previous word (don't use -1 for
 * the first entry!!!)
 */

    2,                /* THANK */
    3,                /* CAN YOU */
    2,                /* CAN I */
    4,-1,             /* YOU ARE, YOU'RE */
    4,                /* I DON'T */
    3,                /* I FEEL */
    3,                /* WHY DON'T YOU */
    2,                /* WHY CAN'T I */
    3,                /* ARE YOU */
    3,                /* I CAN'T */
    4,-1,             /* I AM, I'M */
    2,                /* YOUR */
    5,                /* I WANT */
    9,-1,-1,-1,-1,-1, /* WHAT, HOW, WHO, WHERE, WHEN, WHY */
    2,                /* NAME */
    4,                /* CAUSE */
    4,                /* SORRY */
    4,                /* DREAM */
    1,-1,             /* HELLO, HI */
    5,                /* MAYBE */
    5,-1,-1,          /* NO, NOT, NOTHING */
    3,                /* YOU */
    4,                /* ALWAYS */
    3,                /* THINK */
    6,                /* ALIKE */
    4,                /* YES */
    6,                /* FRIEND */
    7,                /* COMPUTER */
    7                 /* NOKEYFOUND */
};

int *first_response;
int *current_response;
int *last_response;


int n1,  /* Number of elements in keywords (minus 1) */
    n2,  /* Number of elements in wordin and wordout (minus 1) */
    key;

int keyword_size, num_of_responses_size, wordin_size, wordout_size;

/**************************************************************
 function init
 Initializes some critical variables
 **************************************************************/
void init(void)
{
    int loop, response_size, response_number = 0, number = 0;
    int num_of_responses_size;

    /* Set n1 and n2 and some sizes */
    num_of_responses_size = sizeof(num_of_responses) / sizeof(int);
    keyword_size = sizeof(keyword) / sizeof(int);
    wordin_size = sizeof(wordin) / sizeof(char *);
    wordout_size = sizeof(wordout) / sizeof(char *);
    n1 = keyword_size - 1;
    n2 = wordin_size - 1;
    if (keyword_size != num_of_responses_size)
    {
        printf("The sizes of the arrays \"keyword\" and \"num_of_responses\"\n");
        printf("do not match!\n\n");
        printf("sizeof(keyword) = %li, sizeof(num_of_responses) = %li\n",
                keyword_size, num_of_responses_size);
        exit (11);
    }

    if (wordin_size != wordout_size)
    {
        printf("The sizes of the arrays \"wordin\" and \"wordout\"\n");
        printf("do not match!\n\n");
        printf("sizeof(wordin) = %li, sizeof(wordout) = %li\n",
                wordin_size, wordout_size);
        exit (11);
    }

    /* Allocate the memory for first_response, current_response, and last_response */
    response_size = sizeof(num_of_responses);
    if ((first_response = (int *)malloc(response_size)) == NULL)
    {
        printf("Unable to allocate sufficient memory\n");
        exit (12);
    }

    if ((current_response = (int *)malloc(response_size)) == NULL)
    {
        printf("Unable to allocate sufficient memory\n");
        exit (12);
    }

    if ((last_response = (int *)malloc(response_size)) == NULL)
    {
        printf("Unable to allocate sufficient memory\n");
        exit (12);
    }

    /* Set first_response, current_response, and last_response */
    for (loop = 0; loop < num_of_responses_size; loop++)
    {
        if (num_of_responses[loop] == -1)
        {
            response_number -= number;  /* Synonym of last word */
        }
        else
        {
            number = num_of_responses[loop];
        }

        first_response[loop] = response_number;
        current_response[loop] = response_number;
        last_response[loop] = response_number + number - 1;
        response_number += number;
    }

    /* Print the GNU license */
    puts("Eliza - The classic artificial intelligence therapist");
    puts("Creative Computing");
    puts("Morristown, New Jersey");
    puts("Adapted for AMiGA (in BASIC) by Patricia Danielson and Paul Hashfield");
    puts("Ported to C for maximum compatibility by James Williams");
    puts("Copyright (C) 1995  James Williams");
    puts("");
    puts("This program is free software; you can redistribute it and/or modify");
    puts("it under the terms of the GNU General Public License as published by");
    puts("the Free Software Foundation; either version 2 of the License, or");
    puts("(at your option) any later version.");
    puts("");
    puts("This program is distributed in the hope that it will be useful,");
    puts("but WITHOUT ANY WARRANTY; without even the implied warranty of");
    puts("MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the");
    puts("GNU General Public License for more details.");
    puts("");
    puts("You should have received a copy of the GNU General Public License");
    puts("along with this program; if not, write to the Free Software");
    puts("Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.");
    puts("");

    /* That should be it */
}


/**************************************************************
 function exchange_words
 Given a string, position and length of a word to swap out, and
 a new word to swap in, this function does the swap
 **************************************************************/
void exchange_words(char *string, int word_start, int word_len,
                    char *new_word)
{
    /* This function makes the assumption that all the values given to
     * it are valid.
     */

    char temp_storage[MAX_BUFFER];

#ifdef DEBUG
char cnum;

printf("  exchanging [");
for (cnum = word_start; cnum < word_start+word_len; cnum++)
printf("%c", *(string+cnum));

printf("] with [%s]\n", new_word);
#endif

    strncpy(temp_storage, string, word_start);
    temp_storage[word_start] = '\0';
    strcat(temp_storage, new_word);
    strcat(temp_storage, string+word_start+word_len);
    strcpy(string, temp_storage);
}

/**************************************************************
 function ucase
 Converts a string to all uppercase
 **************************************************************/
void ucase(char *string)
{
    while (*string != '\0')
    {
        *string = toupper(*string);
        string++;
    }
}

/**************************************************************
 function reformat_input
 Converts the input to upper case, removes punctuation, and
 strips redundant white space
 **************************************************************/
void reformat_input(void)
{
    int loop;

    /* First, convert it to uppercase */
    ucase (input_string);

    /* Second, strip punctuation */
    for (loop = 0; loop < strlen(input_string); loop++)
    {
        while (!isalnum(input_string[loop]) && input_string[loop] != '\'' &&
               input_string[loop] != ' ' && input_string[loop] != '\0')
        {
            memmove(input_string+loop, input_string+loop+1,
                    strlen(input_string)-loop);
        }
    }

    /* Third, strip redundant white spaces (spaces and tabs) */
    for (loop = 0; loop < strlen(input_string); loop++)
    {
        while (isws(input_string[loop]) && isws(input_string[loop+1]))
        {
            exchange_words(input_string, loop, 2, " ");
        }
    }

    /* Fourth, trim the string */
    loop = 0;

    while (isws(input_string[loop]))
    {
        loop++;
    }

    memmove(input_string, input_string+loop, strlen(input_string)-loop+1);

    loop = strlen(input_string) - 1;

    while (isws(input_string[loop]) && input_string[loop] != '\0')
    {
        loop--;
    }

    input_string[loop+1] = '\0';
}


/**************************************************************
 function get_input
 This function reads a line from the user
 returns whether or not the line is useable
 **************************************************************/
int get_input(void)
{
    strcpy(input_string_buffer, "  ");
    memset(input_string, '\0', MAX_BUFFER);
    printf("> ");
    fgets(input_string, MAX_BUFFER, stdin);
    reformat_input();
#ifdef DEBUG
printf("[%s]\n", input_string);
#endif

    /* See if the user told Eliza to shutup (thereby quitting the program */
    if (strstr(input_string, "SHUT"))
    {
        printf("O.K. IF YOU FEEL THAT WAY I'LL SHUT UP....\n");
        exit (0);
    }

    if (strlen(input_string) == 0)
    {
        printf("I BEG YOUR PARDON?\n");
        return 1;
    }

    if (strcmp(input_string, last_input_string) == 0)
    {
        printf("PLEASE DON'T REPEAT YOURSELF!\n");
        return 1;  /* Unuseable */
    }

    strcpy(last_input_string, input_string);
    strcat(input_string, "  ");  /* Add two spaces to the end of the string */

    return 0;  /* Useable */
}


/**************************************************************
 function find_keyword
 Searches the input string for a known keyword
 **************************************************************/
char *find_keyword(void)
{
    static char found[MAX_BUFFER];

    found[0] = '\0';  /* Empty out the string */

    for (key = 0; key <= n1; key++)
    {
        if (strstr(input_string_buffer, keyword[key]))
        {
            /* Keyword has been identified */
            strcpy (found, keyword[key]);
            break;
        }
    }

    if (key > n1)
    {
        key = n1;  /* In case no key was found */
    }

    return found;
}

/**************************************************************
 function conjugate
 Takes a part of the input string and conjugates it using the
 list of strings to be swapped.
 **************************************************************/
char *conjugate(char *found)
{
    static char tail[MAX_BUFFER];
    char *tail_ptr;
    char string_pos;
    int word;

#ifdef DEBUG
puts("Inside conjugate");
puts("  Extracting end of input string");
#endif
    /* Extract the end of the input string */
    tail_ptr = strstr(input_string_buffer, found) + strlen(found);
    strcpy(tail, " ");
    strcat(tail, tail_ptr);
    strcat(tail, " ");

#ifdef DEBUG
printf("  tail = [%s]\n", tail);
printf("  strlen(tail) = %i\n", strlen(tail));
puts("  Swapping first and second person phrases");
#endif
    /* swap first and second person phrases */
    for (string_pos = 0; string_pos < strlen(tail); string_pos++)
    {
        for (word = 0; word <= n2; word++)
        {
            tail_ptr = tail + string_pos;

            /* Search through the string for phrases character by character */
            if (strncmp(wordin[word], tail_ptr, strlen(wordin[word])) == 0)
            {
                exchange_words(tail, string_pos, strlen(wordin[word]), wordout[word]);
                string_pos = string_pos + strlen(wordout[word]) - 2;
                break;
            }
        }
    }

    return tail;
}


/**************************************************************
 function response
 Builds and prints a response to the user input
 **************************************************************/
void response(char *tail)
{
    char output[MAX_BUFFER];

    strcpy (output, replies[current_response[key]]);
    current_response[key]++;

    if (current_response[key] > last_response[key])
    {
        current_response[key] = first_response[key];
    }

    if (output[strlen(output) - 1] != '*')
    {
        puts(output);
        return;
    }

    if (strcmp(tail, "  ") == 0)
    {
        puts("YOU WILL HAVE TO ELABORATE MORE FOR ME TO HELP YOU\n");
        return;
    }

    output[strlen(output)-1] = '\0';
    printf("%s%s\n", output, tail);
}

/**************************************************************/

main ()
{
    char *found_keyword, *tail;

    init();
    printf("HI!  I'M ELIZA.  WHAT'S YOUR PROBLEM?\n");

    for (;;)  /* An infinte loop, I'm afraid */
    {
#ifdef DEBUG
puts("Before reading input");
#endif
        if (get_input() != 0)
        {
            continue;  /* Unuseable input - try again */
        }
#ifdef DEBUG
puts("Before searching for keyword");
#endif
        found_keyword = find_keyword();
#ifdef DEBUG
puts("After searching for keyword");
#endif

        if (found_keyword[0] != '\0')
        {
#ifdef DEBUG
printf("Found keyword [%s] - conjugating\n", found_keyword);
#endif
            tail = conjugate(found_keyword);
#ifdef DEBUG
puts("Done conjugating");
#endif
        }
        else
        {
            tail = "";
        }

#ifdef DEBUG
puts("Building response");
#endif
        response(tail);
#ifdef DEBUG
puts("Done building response");
#endif
    }
}
